<?php
/**
 * Account Lock
 * https://webenginecms.org/
 * 
 * @version 1.2.0
 * @author Lautaro Angelico <http://lautaroangelico.com/>
 * @copyright (c) 2013-2020 Lautaro Angelico, All Rights Reserved
 * @build w3c8c718b75a0f1fa1a557f7f9d70877
 */

namespace Plugin\AccountLock;

class AccountLock {
	
	private $_table;
	private $_accountColumn;
	private $_securityCodeColumn = 'SecCode';
	private $_defaultSecurityCode = 0;
	
	private $_configXml = 'config.xml';
	private $_modulesPath = 'modules';
	private $_emailsPath = 'emails';
	
	private $_user;
	private $_key;
	private $_timestamp;
	private $_hash;
	
	private $_salt = 'WebEngineCMS';
	private $_linkExpirationTime = 300;
	
	private $_usercpmenu = array(
		array(
			'active' => true,
			'type' => 'internal',
			'phrase' => 'accountlock_title',
			'link' => 'accountlock',
			'icon' => 'usercp_default.png',
			'visibility' => 'user',
			'newtab' => false,
			'order' => 999,
		),
	);
	
	// CONSTRUCTOR
	
	function __construct() {
		// load databases
		$this->common = new \common();
		$this->mu = \Connection::Database('MuOnline');
		
		// config file path
		$this->configFilePath = __PATH_ACCOUNTLOCK_ROOT__.$this->_configXml;
		if(!file_exists($this->configFilePath)) throw new \Exception(lang('accountlock_error_2', true));
		$xml = simplexml_load_file($this->configFilePath);
		if(!$xml) throw new \Exception(lang('accountlock_error_2', true));
		$this->_configs = convertXML($xml->children());
		if(!is_array($this->_configs)) throw new \Exception(lang('accountlock_error_2', true));
		
		// set configs
		$this->_linkExpirationTime = $this->_configs['link_expiration_time'];
		
		// tables
		$this->_table = _TBL_AC_;
		$this->_accountColumn = _CLMN_AC_ID_;
	}
	
	// PUBLIC FUNCTIONS
	
	public function loadModule($module) {
		if(!\Validator::Alpha($module)) throw new \Exception(lang('accountlock_error_4', true));
		if(!$this->_moduleExists($module)) throw new \Exception(lang('accountlock_error_4', true));
		if(!@include_once(__PATH_ACCOUNTLOCK_ROOT__ . $this->_modulesPath . '/' . $module . '.php')) throw new \Exception(lang('accountlock_error_4', true));
	}
	
	public function setUser($user) {
		$this->_user = $user;
	}
	
	public function setKey($key) {
		$this->_key = $key;
	}
	
	public function setTimestamp($timestamp) {
		$this->_timestamp = $timestamp;
	}
	
	public function setHash($hash) {
		$this->_hash = $hash;
	}
	
	public function lock() {
		
		// check if account is set
		if(!check_value($this->_user)) throw new \Exception(lang('accountlock_error_3',true));
		
		// load accountcharacter data
		$accountCharacterData = $this->_getAccountCharacterData();
		if(!is_array($accountCharacterData)) throw new \Exception(lang('accountlock_error_6',true));
		
		// check if already locked
		if($accountCharacterData[$this->_securityCodeColumn] != $this->_defaultSecurityCode) throw new \Exception(lang('accountlock_error_7',true));
		
		// generate lock key
		$securityKey = $this->_generateSecurityKey();
		$this->setKey($securityKey);
		
		// lock account
		$lock = $this->_lockAccount();
		if(!$lock) throw new \Exception(lang('accountlock_error_10',true));
		
		// success redirect
		redirect(1, 'accountlock/');
	}
	
	public function requestUnlockEmail() {
		
		// check if account is set
		if(!check_value($this->_user)) throw new \Exception(lang('accountlock_error_3',true));
		
		// load accountcharacter data
		$accountCharacterData = $this->_getAccountCharacterData();
		if(!is_array($accountCharacterData)) throw new \Exception(lang('accountlock_error_6',true));
		
		// check if already unlocked
		if($accountCharacterData[$this->_securityCodeColumn] == $this->_defaultSecurityCode) throw new \Exception(lang('accountlock_error_8',true));
		
		// unlock link
		$unlockLink = $this->_generateUnlockLink($this->_user, $accountCharacterData[$this->_securityCodeColumn]);
		
		// load account info
		$accountId = $this->common->retrieveUserID($accountCharacterData[$this->_accountColumn]);
		if(!check_value($accountId)) throw new \Exception(lang('accountlock_error_6',true));
		$accountInfo = $this->common->accountInformation($accountId);
		if(!is_array($accountInfo)) throw new \Exception(lang('accountlock_error_6',true));
		
		// has email address
		if(!check_value($accountInfo[_CLMN_EMAIL_])) throw new \Exception(lang('accountlock_error_13',true));
		
		// send unlock email
		try {
			
			$email = new \Email();
			$email->setSubject(lang('accountlock_email_1_title'));
			$email->setCustomTemplate(__PATH_ACCOUNTLOCK_ROOT__ . $this->_emailsPath . '/PLUGIN_ACCOUNTLOCK.txt');
			$email->addVariable('{USERNAME}', $accountCharacterData[$this->_accountColumn]);
			$email->addVariable('{UNLOCK_LINK}', $unlockLink);
			$email->addAddress($accountInfo[_CLMN_EMAIL_]);
			$email->send();
			
		} catch(\Exception $ex) {
			throw new \Exception(lang('accountlock_error_12',true));
		}
		
		// success redirect
		redirect(1, 'accountlock/?email=sent');
		
	}
	
	public function unlock() {
		
		// check if account is set
		if(!check_value($this->_user)) throw new \Exception(lang('accountlock_error_3',true));
		
		// check if key is set
		if(!check_value($this->_key)) throw new \Exception(lang('accountlock_error_5',true));
		
		// load accountcharacter data
		$accountCharacterData = $this->_getAccountCharacterData();
		if(!is_array($accountCharacterData)) throw new \Exception(lang('accountlock_error_6',true));
		
		// check if already unlocked
		if($accountCharacterData[$this->_securityCodeColumn] == $this->_defaultSecurityCode) throw new \Exception(lang('accountlock_error_8',true));
		
		// compare lock key
		if($this->_key != $accountCharacterData[$this->_securityCodeColumn]) throw new \Exception(lang('accountlock_error_9',true));
		
		// check timestamp
		if($this->_isTimestampExpired()) throw new \Exception(lang('accountlock_error_11',true));
		
		// validate hash
		if(!$this->_validateHash()) throw new \Exception(lang('accountlock_error_11',true));
		
		// unlock account
		$unlock = $this->_unlockAccount();
		if(!$unlock) throw new \Exception(lang('accountlock_error_10',true));
		
		// success redirect
		redirect(1, 'accountlock/');
	}
	
	public function isLocked() {
		if(!check_value($this->_user)) return;
		$accountCharacterData = $this->_getAccountCharacterData();
		if(!is_array($accountCharacterData)) return;
		if($accountCharacterData[$this->_securityCodeColumn] == $this->_defaultSecurityCode) return;
		return true;
	}
	
	public function checkPluginUsercpLinks() {
		if(!is_array($this->_usercpmenu)) return;
		$cfg = loadConfig('usercp');
		if(!is_array($cfg)) return;
		foreach($cfg as $usercpMenu) {
			$usercpLinks[] = $usercpMenu['link'];
		}
		foreach($this->_usercpmenu as $pluginMenuLink) {
			if(in_array($pluginMenuLink['link'],$usercpLinks)) continue;
			$cfg[] = $pluginMenuLink;
		}
		usort($cfg, function($a, $b) {
			return $a['order'] - $b['order'];
		});
		$usercpJson = json_encode($cfg, JSON_PRETTY_PRINT);
		$cfgFile = fopen(__PATH_CONFIGS__.'usercp.json', 'w');
		if(!$cfgFile) throw new \Exception('There was a problem opening the usercp file.');
		fwrite($cfgFile, $usercpJson);
		fclose($cfgFile);
	}
	
	// PRIVATE FUNCTIONS
	
	private function _moduleExists($module) {
		if(!check_value($module)) return;
		if(!file_exists(__PATH_ACCOUNTLOCK_ROOT__ . $this->_modulesPath . '/' . $module . '.php')) return;
		return true;
	}
	
	private function _generateSecurityKey() {
		return rand(111111,999999);
	}
	
	private function _lockAccount() {
		if(!check_value($this->_user)) return;
		if(!check_value($this->_key)) return;
		$result = $this->mu->query("UPDATE ".$this->_table." SET ".$this->_securityCodeColumn." = ? WHERE ".$this->_accountColumn." = ?", array($this->_key, $this->_user));
		if(!$result) return;
		return true;
	}
	
	private function _unlockAccount() {
		if(!check_value($this->_user)) return;
		$result = $this->mu->query("UPDATE ".$this->_table." SET ".$this->_securityCodeColumn." = ? WHERE ".$this->_accountColumn." = ?", array($this->_defaultSecurityCode, $this->_user));
		if(!$result) return;
		return true;
	}
	
	private function _getAccountCharacterData() {
		if(!check_value($this->_user)) return;
		$result = $this->mu->query_fetch_single("SELECT * FROM ".$this->_table." WHERE ".$this->_accountColumn." = ?", array($this->_user));
		if(!is_array($result)) return;
		return $result;
	}
	
	private function _generateHash($username, $key, $timestamp) {
		return md5($username . $key . $timestamp . $this->_salt);
	}
	
	private function _generateUnlockLink($username, $key) {
		$timestamp = time();
		$hash = $this->_generateHash($username,$key,$timestamp);
		$unlockLink = __BASE_URL__ . 'accountlock/unlock/user/'.$username.'/key/'.$key.'/t/'.$timestamp.'/hash/' . $hash;
		return $unlockLink;
	}
	
	private function _validateHash() {
		if(!check_value($this->_user)) return;
		if(!check_value($this->_key)) return;
		if(!check_value($this->_timestamp)) return;
		if(!check_value($this->_hash)) return;
		$hash = $this->_generateHash($this->_user,$this->_key,$this->_timestamp);
		if($hash != $this->_hash) return;
		return true;
	}
	
	private function _isTimestampExpired() {
		if(!check_value($this->_timestamp)) return true;
		if(time() > ($this->_timestamp+$this->_linkExpirationTime)) return true;
		return;
	}
	
}